### index.janet
###
### Bagatto SSG site for bear-blog like site
### Bagatto port by hdrz

#
# Global Config
#
## `bagatto/set-output-dir!` will set the top level of the directory
## hierarchy that files are created in. This is quite useful, as it
## allows us to use relative directories elsewhere when linking within
## the site.
(bagatto/set-output-dir! "site")
(def base-url "https://hdrz.github.io/bagatto-bearblog/")

# add :slug, :url to the attrs
(defn parse-post
 [src attrs]
 (let [attrs (bagatto/parse-mago src attrs)]
  (put attrs :slug (bagatto/slug-from-attr attrs :title))
  (put attrs :url (string (attrs :slug) ".html"))))

# Now we will link all tags to relevant posts,
# so that we can build the tag filtering thing.
# For each tag we link it to the posts in which
# it appears.
(defn parse-tags
 "get all the tags and link them to posts"
 [src attrs]
 (def tags @{})
 (let [attrs (parse-post src attrs)]
  (if (in attrs :tags)
   (let [postvars {:title (attrs :title)
                   :url (attrs :url)
                   :date (attrs :date)}]
    (each newtag (attrs :tags)
     # insert new tag and related post
     (put tags newtag postvars)))))
 tags)

# Sort the tags into one array
(defn sort-tags
 "Sort tags into one array"
 [tags]
 (def newtags @{})
 (each tab tags
  (eachp [key val] tab
    (if (in newtags key)
     # tag already exists, add post data to tag
     (array/push (get newtags key) val)
     # insert new tag and related post
     (put newtags key @[val]))))
 # now transform to array
 (def tagarray @[])
 (eachp [key val] newtags
  (array/push tagarray {:tag key :slug (bagatto/slugify key) :posts val}))
 # return sorted array
 (sort tagarray (fn [a b] (< (a :tag) (b :tag)))))

#
# Data Specification
#
### The first value we need to define is `data`, which is the *data
### specification* for our site. It is a simple key-value data
### structure which maps specification names to specifications. Each
### data spec contains an `attrs` and an optional `src` value. If it
### only contains `attrs`, the value at `attrs` will be included
### directly into the site data. If it contains `src`, the value at
### `src` will be evaluated to load one or more *source files*, and
### the value at `attrs` will be treated as a function that's applied
### to each source file, and expected to return the metadata used in
### rendering the site.

(def data {:config {:attrs {:title "Bagatto bear-blog"
                    :description "bear-blog like site generated by Bagatto"
                    :author "hdrz"
                    :baseurl base-url  # used in `partials/head.temple` for <base> tag
                    :menus [{:name "Home" :url ""}
                            {:name "Bear" :url "bear.html"}
                            {:name "blog" :url "blog/"}
                            {:name "tags" :url "blog/tags/"}
                            {:name "README⤴" :url "../README.md"}]
                    :static "static/"
                    :images "static/images/"
                    :showMadeWithLine true  # used in the footer
                    :baggatoBearUrl "https://hdrz.cc/museum/bagatto-bearblog"
                    :keywords ["bagatto-bearblog" "blog" "janet"]}}
           #
           # all the pages
           :pages {:src (bagatto/slurp-* "content/*.md")
                   :attrs bagatto/parse-mago}
           #
           # all the posts
           :posts {:src (bagatto/slurp-* "content/posts/*.md")
                   :attrs parse-post
                   :transform (bagatto/attr-sorter :date :desc)}
           #
           # all posts tags
           :tags {:src (bagatto/slurp-* "content/posts/*.md")
                  :attrs parse-tags
                  :transform sort-tags}
           #
           # enumerate static content
           :static {:src (bagatto/* "static/*") :attrs bagatto/parse-base}
           :images {:src (bagatto/* "static/images/*") :attrs bagatto/parse-base}
          }
)


# --- PEG Grammar for Replacement ---
(def headings-grammar
 ~{:main (* :hopen :htext)
   :hlevel (<- (* "h"  (range "23")))
   :hopen  (* "<" :hlevel ">")
   :htext  (<- (any (if-not "<" 1)))})

# do the heading replacements
(defn add-heading-ids
 [htmlstring]
 (peg/replace-all
    headings-grammar
    # Define a replacement for every time the ':heading' rule matches.
    # This function receives the captures from the rule as arguments.
    # '_' is the full string matched by the rule, which we ignore.
    (fn [_ tagname text]
      # Add a new id attribute to the opening tag, add anchor
      (def slug (bagatto/slugify text))
      (string/format `<%s id="%s"><a href="#%s">%s</a>`
        tagname
        slug
        slug
        text))
    htmlstring))

# minify the resulting html
(defn minify
 [htmlstring]
 (sh/$< echo ,(string htmlstring) | minify --type html --html-keep-quotes --html-keep-end-tags --html-keep-document-tags))

(defn html-renderer
 [template &opt args]
 # use this line for original html
 #(comp add-heading-ids (bagatto/renderer template args)))
 # use this line for minified version
 (comp minify add-heading-ids (bagatto/renderer template args)))


#
# Site Specification
#
## The second value we need to define is `site`, which will be the
## *site specification* for our site. A site specification maps
## specification names to individual specification entries. Each site
## spec will define one or more files to write by giving the path and
## contents of the new files. The names are not used by Bagatto, but
## are useful to the site author for organizational purposes.
#
# We use {:relpath "relative/menu/path"} in order to render the
# nav menu links, since different pages are nested differently.
# this way all the links on the rendered site are relative.

(def site {#
           # render SEO stuff
           #
           :rssfeed {:dest "feed.xml"
                     :out (bagatto/renderer "/templates/feed")}
           #
           :sitemap {:dest "sitemap.xml"
                     :out (bagatto/renderer "/templates/sitemap")}
           #
           :robots {:dest "robots.txt"
                    :out (bagatto/renderer "/templates/robots")}
           #
           # render the pages (home, bear, ...)
           #
           :pages {:each :pages
                   :dest (bagatto/%p '%i :slug '% ".html")
                   :out (bagatto/renderer "/templates/page" {:relpath "./"})}
           #
           # render the posts
           #
           :indexposts {:dest "blog/index.html"
                        :out (bagatto/renderer "/templates/postlist" {:relpath "../"})}
           #
           :posts {:each :posts
                   :dest (bagatto/%p "blog" '%i :slug '% ".html")
                   :out (html-renderer "/templates/post" {:relpath "../"})}
           #
           # render the tags
           #
           :indextags {:dest "blog/tags/index.html"
                       :out (bagatto/renderer "/templates/taglist" {:relpath "../../"})}
           #
           :tags {:each :tags
                  :dest (bagatto/%p "blog/tags" '%i :slug '% ".html")
                  :out (bagatto/renderer "/templates/tag" {:relpath "../../"})}
           #
           # Copy all static content to generated site
           :static {:each :static :dest (bagatto/path-copier "static")}
           :images {:each :images :dest (bagatto/path-copier "static/images")}
          }
)
